import wx
from gui import guiHelper
import addonHandler
import globalPluginHandler
import globalVars
import gui
from collections import defaultdict
import languageHandler
import ui
from synthDrivers.vocalizer_expressive2 import _config, languageDetection, ve2, getResourcePaths, getAvailableResources

addonHandler.initTranslation()

class VocalizerLanguageSettingsDialog(gui.SettingsDialog):
	title = _("Vocalizer Automatic Language Switching Settings")

	def __init__(self, parent):
		self._localeToVoices = {}

		with ve2.preOpenVocalizer(getResourcePaths()) as success:
			if success:
				for l, v in getAvailableResources().items():
					self._localeToVoices[l.id] = v
					if "_" in l.id:
						lang = l.id.split("_")[0]
						if lang not in self._localeToVoices:
							self._localeToVoices[lang] = []
						self._localeToVoices[lang].extend(v)

		self._dataToPercist = defaultdict(lambda: {})
		self._locales = sorted([l for l in self._localeToVoices if len(self._localeToVoices[l]) > 0])
		latinSet = set(languageDetection.ALL_LATIN) & set(l for l in self._locales if len(l) == 2)
		self._latinLocales = sorted(list(latinSet))
		CJKSet = set(languageDetection.CJK) & set(l for l in self._locales if len(l) == 2)
		self._CJKLocales = sorted(list(CJKSet))
		super(VocalizerLanguageSettingsDialog, self).__init__(parent)

	def makeSettings(self, sizer):
		settingsSizerHelper = guiHelper.BoxSizerHelper(self, sizer=sizer)

		helpLabel = wx.StaticText(self, label=_("Select a language, and then configure the voice to be used:"))
		helpLabel.Wrap(self.GetSize()[0])
		settingsSizerHelper.addItem(helpLabel)

		localeNames = list(map(self._getLocaleReadableName, self._locales))
		self._localesChoice = settingsSizerHelper.addLabeledControl(_("Locale Name:"), wx.Choice, choices=localeNames)
		self.Bind(wx.EVT_CHOICE, self.onLocaleChanged, self._localesChoice)

		self._voicesChoice = settingsSizerHelper.addLabeledControl(_("Voice Name:"), wx.Choice, choices=[])
		self.Bind(wx.EVT_CHOICE, self.onVoiceChange, self._voicesChoice)

		self._useUnicodeDetectionCheckBox = wx.CheckBox(self,
		# Translators: Wether to use or not unicode characters based language detection.
			label=_("Detect text language based on unicode characters"))
		self._useUnicodeDetectionCheckBox.SetValue(_config.vocalizerConfig["autoLanguageSwitching"]["useUnicodeLanguageDetection"])
		settingsSizerHelper.addItem(self._useUnicodeDetectionCheckBox)

		self._ignoreNumbersCheckBox = wx.CheckBox(self,
		# Translators: Either to ignore or not numbers when language detection is active
		label=_("Ignore numbers when detecting text language"))
		self._ignoreNumbersCheckBox.SetValue(_config.vocalizerConfig["autoLanguageSwitching"]["ignoreNumbersInLanguageDetection"])
		settingsSizerHelper.addItem(self._ignoreNumbersCheckBox)

		self._ignorePunctuationCheckBox = wx.CheckBox(self,
		# Translators: Either to ignore or not ASCII punctuation when language detection is active
		label=_("Ignore common punctuation when detecting text language"))
		self._ignorePunctuationCheckBox.SetValue(_config.vocalizerConfig["autoLanguageSwitching"]["ignorePunctuationInLanguageDetection"])
		settingsSizerHelper.addItem(self._ignorePunctuationCheckBox)

		latinChoiceLocaleNames = [self._getLocaleReadableName(l) for l in self._latinLocales]
		self._latinChoice = settingsSizerHelper.addLabeledControl(_("Language assumed for latin characters:"), wx.Choice, choices=latinChoiceLocaleNames)
		latinLocale = _config.vocalizerConfig["autoLanguageSwitching"]["latinCharactersLanguage"]
		try:
			self._latinChoice.Select(self._latinLocales.index(latinLocale))
		except ValueError:
			self._latinChoice.Select(0)
		if not latinChoiceLocaleNames:
			self._latinChoice.Disable()

		CJKChoiceLocaleNames = [self._getLocaleReadableName(l) for l in self._CJKLocales]
		self._CJKChoice = settingsSizerHelper.addLabeledControl(_("Language assumed for CJK characters:"), wx.Choice, choices=CJKChoiceLocaleNames)
		CJKLocale = _config.vocalizerConfig["autoLanguageSwitching"]["CJKCharactersLanguage"]
		try:
			self._CJKChoice.Select(self._CJKLocales.index(CJKLocale))
		except ValueError:
			self._CJKChoice.Select(0)
		if not CJKChoiceLocaleNames:
			self._CJKChoice.Disable()

	def postInit(self):
		self._updateVoicesSelection()
		self._localesChoice.SetFocus()

	def _updateVoicesSelection(self):
		localeIndex = self._localesChoice.GetCurrentSelection()
		if localeIndex < 0:
			self._voicesChoice.SetItems([])
		else:
			locale = self._locales[localeIndex]
			voices = sorted([v.id for v in self._localeToVoices[locale]])
			self._voicesChoice.SetItems(voices)
			if locale in _config.vocalizerConfig["autoLanguageSwitching"]:
				voice = _config.vocalizerConfig["autoLanguageSwitching"][locale]["voice"]
				if voice:
					self._voicesChoice.SetStringSelection(voice)

	def onLocaleChanged(self, event):
		self._updateVoicesSelection()

	def onVoiceChange(self, event):
		localeIndex = self._localesChoice.GetCurrentSelection()
		if localeIndex >= 0:
			locale = self._locales[localeIndex]
			self._dataToPercist[locale]["voice"] = self._voicesChoice.GetStringSelection()
		else:
			self._dataToPercist[locale]["voice"] = None

	def onOk(self, event):
		# Update Configuration
		for locale in self._dataToPercist:
			_config.vocalizerConfig["autoLanguageSwitching"][locale] = self._dataToPercist[locale]

		_config.vocalizerConfig["autoLanguageSwitching"]["useUnicodeLanguageDetection"] = self._useUnicodeDetectionCheckBox.GetValue()
		_config.vocalizerConfig["autoLanguageSwitching"]["ignoreNumbersInLanguageDetection"] = self._ignoreNumbersCheckBox.GetValue()
		_config.vocalizerConfig["autoLanguageSwitching"]["ignorePunctuationInLanguageDetection"] = self._ignorePunctuationCheckBox.GetValue()
		if self._latinChoice.IsEnabled():
			_config.vocalizerConfig["autoLanguageSwitching"]["latinCharactersLanguage"] = self._latinLocales[self._latinChoice.GetCurrentSelection()]
		if self._CJKChoice.IsEnabled():
			_config.vocalizerConfig["autoLanguageSwitching"]["CJKCharactersLanguage"] = self._CJKLocales[self._CJKChoice.GetCurrentSelection()]
		super(VocalizerLanguageSettingsDialog, self).onOk(event)

	def _getLocaleReadableName(self, locale):
		description = languageHandler.getLanguageDescription(locale)
		return "%s - %s" % (description, locale) if description else locale

class GlobalPlugin(globalPluginHandler.GlobalPlugin):
	scriptCategory = _("Vocalizer expressive 2")

	def __init__(self):
		super(GlobalPlugin, self).__init__()
		if globalVars.appArgs.secure:
			return
		# See if we have at least one voice installed
		if getResourcePaths():
			self.menuItem = gui.mainFrame.sysTrayIcon.preferencesMenu.Append(wx.ID_ANY, _("Vocalizer automatic &language switching settings..."), _("Configure which voice is used for each language."))
			gui.mainFrame.sysTrayIcon.Bind(wx.EVT_MENU , lambda e : gui.mainFrame._popupSettingsDialog(VocalizerLanguageSettingsDialog), self.menuItem)

	def terminate(self):
		try:
			gui.mainFrame.sysTrayIcon.preferencesMenu.Remove(self.menuItem)
		except Exception:
			pass

	def script_toggleAutomaticLanguageSwitching(self, gesture):
		newState = not _config.vocalizerConfig["autoLanguageSwitching"]["useUnicodeLanguageDetection"]
		_config.vocalizerConfig["autoLanguageSwitching"]["useUnicodeLanguageDetection"] = newState
		if newState:
			ui.message(_("Automatic language switching is enabled"))
		else:
			ui.message(_("Automatic language switching is disabled"))
	script_toggleAutomaticLanguageSwitching.__doc__ = _("Toggles automatic language switching")
